/* ============ */
/* ksmirnov.c	*/
/* ============ */
#include <math.h>
#include <mconf.h>
extern	double	MINLOG;
/* ==================================================================== */
/* KSmirnov - Calculates Kolmogorov-Smirnov Probability for Given n & e */
/* ==================================================================== */
double
KSmirnov(int n, double e)
{
    int     v, nn;
    double  evn, omevn, p, t, c, lgamnp1, RetVal;

/* ------------------------------------------------------------ *
 * Exact Smirnov statistic, for one-sided test [see ref].	*
 *								*
 *		    [n(1-e)]					*
 *	+	       -		v-1		 n-v	*
 *  Pr{D <= e} = 1 - e >  C   (e + v/n)    (1 - e - v/n)	*
 *	n	       - n v					*
 *		      v=0					*
 *								*
 *  This is the probability that the statistic Dn <= e. 	*
 *								*
 *  [n(1-e)] is the largest integer not exceeding n(1-e).	*
 *								*
 *  nCv is the number of combinations of n things taken v	*
 *	at a time (binomial coefficient).			*
 *								*
 *  Reference:							*
 *  Z.W. Birnbaun and Fred H. Tingey, One-Sided Confidence	*
 *  Countours for Probability Distribution Functions, Annals of *
 *  Mathematical Statistics 22(1951), pp 592-596		*
 *								*
 *  Coded by: Stephen L. Moshier, October 1995			*
 *  Modified by: K. B. Williams, December 1995			*
 * ------------------------------------------------------------ */

    if (n <= 0 || e <= 0.0 || e > 1.0)
    {
	if (e == 0)
	{
	    RetVal = 0;
	}
	else
	{
	    char    *ErrPrt;

	    if (n <= 0)
	    {
		ErrPrt = "KSmirnov (n <= 0)";
	    }
	    else if (e < 0)
	    {
		ErrPrt = "KSmirnov (e < 0)";
	    }
	    else
	    {
		ErrPrt = "KSmirnov (e > 1)";
	    }

	    mtherr(ErrPrt, DOMAIN);
	    RetVal = -1;
	}
    }
    else
    {
	nn = (int) floor((double) n * (1.0 - e));
	p = 0.0;
	if (n < 1013)
	{
	    c = 1.0;
	    for (v = 0; v <= nn; v++)
	    {
		evn = e + ((double) v) / n;

		if (evn < 1)
		{
		    t = (double) (v-1) * log(evn) +
			(double) (n-v) * log(1.0-evn);

		    if (t > MINLOG)
		    {
			p += c * exp(t);
		    }
		}

		/* Next combinatorial term; worst case error = 4e-15.  */
		c *= ((double) (n - v)) / (v + 1);
	    }
	}
	else
	{
	    lgamnp1 = lgam((double) (n + 1));
	    for (v = 0; v <= nn; v++)
	    {
		evn = e + ((double) v) / n;
		omevn = 1.0 - evn;
		if (fabs(omevn) > 0.0)
		{
		    t = lgamnp1
		      - lgam((double) (v + 1))
		      - lgam((double) (n - v + 1))
		      + (v - 1) * log(evn)
		      + (n - v) * log(omevn);
		    if (t > MINLOG)
			p += exp(t);
		}
	    }
	}

	RetVal = 1 - p * e;
    }

    return (RetVal);
}
